# -*- tcl -*-
# Generation (Conversion subsystem - Table of Contents)
# ------------------------------------------------------

package require tools
package require optchecker

namespace eval app-gen-idx {}

# ------------------------------------------------------
# Acceptable syntax for cmdline functionality

set ::app-gen-idx::help(cmdline) {

[call [cmd {@appname@}] [method gen-idx] [opt [arg options...]] [arg format] [arg iomap] [arg idxfile]]

Reads the file [arg idxfile] and writes an index in the output

[arg format] to [const stdout]. Assumes that the [arg idxfile] is
written in the [syscmd docidx] format. The resolution of symbolic to
actual filenames is guided by the [arg iomap]. It is assumed that the
map file is in the same format as generated by the subcommand

[method map].

The conversion can be influenced and manipulated by the options listed
below.

[list_begin definitions]
[lst_item "[option -varstring] [arg varname] [arg string]"]

This option is used to set format specific parameters to some
value. The parameter is specified by [arg varname], the new value by
[arg string].

[nl]

Currently only the HTML engine supports engine parameters.

[lst_item "[option -varfile] [arg varname] [arg file]"]

This option is used to set format specific parameters to some
value. The parameter is specified by [arg varname], the new value by
the contents of the file [arg file].

[nl]

Currently only the HTML engine supports engine parameters.

[lst_item "[option -subst] [arg varname] [arg marker] [arg path]"]

When used multiple times the data accumulates, in contrast to the
other options, where the last occurence is dominant.

[nl]

It declares that the data for the engine parameter [arg varname]
contains a variable link, that all locations for this link are marked
by the string [arg marker], and that the link has to refer to the
output file [arg path]. Based upon this information the formatter
will, for each output file in the [arg iomap], replace the marker with
a proper link to the [arg path], based upon the relative locations of
the current file and the file linked to, i.e. [arg path].

[nl]

An occurence of this option for a variable [arg varname] is ignored if
no engine parameters for [arg varname] were specified via either
[option -varfile] or [option -varstring].

[list_end]
}

proc ::app-gen-idx::help {topic} {
    variable help
    if {[info exists help($topic)]} {return $help($topic)}
    return {}
}

# ------------------------------------------------------
# Implementation of cmdline functionality.

proc ::app-gen-idx::run {argv} {
    set errstring "wrong#args: gen-idx ?-varstring var string? ?-varfile var file? format iomap idxfile"

    if {[llength $argv] < 3} {tools::usage $errstring}

    array set para {}
    set subst {}

    while {[string match -* [lindex $argv 0]]} {
	if {[string equal [lindex $argv 0] -varstring]} {
	    if {[llength $argv] < 6} {tools::usage $errstring}

	    set varname  [lindex $argv 1]
	    set varvalue [lindex $argv 2]
	    set argv     [lrange $argv 3 end]

	    set para($varname) $varvalue
	    continue
	}
	if {[string equal [lindex $argv 0] -varfile]} {
	    if {[llength $argv] < 6} {tools::usage $errstring}

	    set varname  [lindex $argv 1]
	    ::optcheck::infile [lindex $argv 2] "Parameter content file"
	    set varvalue [tools::getfile [lindex $argv 2]]
	    set argv     [lrange $argv 3 end]

	    set para($varname) $varvalue
	    continue
	}
	if {[string equal [lindex $argv 0] -subst]} {
	    if {[llength $argv] < 6} {tools::usage $errstring}

	    set var  [lindex $argv 1]
	    set mark [lindex $argv 2]
	    set link [lindex $argv 3]
	    set argv [lrange $argv 4 end]

	    lappend subst $var [list $mark $link]
	    continue
	}
	tools::usage $errstring
    }

    if {[llength $argv] < 3} {tools::usage $errstring}

    optcheck::infile [set mapfile [lindex $argv 1]] "Mapping file"
    optcheck::infile [set idxfile [lindex $argv 2]] "IDX file"

    set format  [lindex $argv 0]
    set idxdata [tools::getfile $idxfile]
    set iomap   [tools::readmap $mapfile]

    # Now we modify the iomap so that the actual file names resolve to
    # the filename relative to the generated file, which we find by
    # looking for the virtual input file _index_. If no such file is
    # present we do not modify the map.

    # Before doing the above we also integrate the substitution
    # information into the parameters, if they exist.

    array set _ $iomap
    if {[info exists _(_index_)]} {
	if {[llength $subst] > 0} {
	    array set __ {}
	    foreach {var val} $subst {
		foreach {mark link} $val break
		lappend __($var) $mark [set ref [::tools::link $_(_index_) $link]]
	    }
	    foreach {var map} [array get __] {
		if {[info exists para($var)]} {
		    set para($var) [string map $map $para($var)]
		}
	    }
	}

	foreach key [array names _] {
	    if {[string equal _index_ $key]} {continue}
	    set _($key) [tools::link $_(_index_) $_($key)]
	}
	set iomap [array get _]
    }

    package require dtglue
    puts stdout [dtglue::cvtidx $format $idxdata $iomap [array get para] $idxfile]
    return
}

# ------------------------------------------------------
package provide app-gen-idx 0.1
